#!/usr/bin/env python
# coding: utf-8

# LR_calculations
# 
# Performs calculations described in:
#   Morrison G.S., Weber P., Basu N., Puch-Solis R., Randolph-Quinney P.S. (2021).  
#   Calculation of likelihood ratios for inference of biological sex from human skeletal remains.
#   Forensic Science International: Synergy, 3, article 100202. https://doi.org/10.1016/j.fsisyn.2021.100202
# 
# Requires supplementary data file from:
#   Bartholdy B.P., Sandoval E., Hoogland M.L.P., Schrader S.A. (2020).  
#   Getting rid of dichotomous sex estimations: Why logistic regression should be preferred over discriminant function analysis.
#   Journal of Forensic Sciences, 65, 1685–1691. https://doi.org/10.1111/1556-4029.14482
# 
#   Download "jfo14482-sup-0001-tables1.xlsx" and place a copy in the "data" folder
# 
# Requires      Tested version
#   Python        3.8.5
#   pandas        1.3.0
#   numpy         1.20.3
#   scipy         1.7.0
#   scikit-learn  0.24.2


import warnings
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
from scipy.stats import norm, multivariate_normal
from sklearn.linear_model import LogisticRegression
from sklearn.discriminant_analysis import LinearDiscriminantAnalysis
from functions.Cllr import Cllr
from functions.Tippett import plot_tippett


## Options

# which variable to use for univariate models
univar_dim = 0  # choose from: 0, 1
bivar_str = ['HeadD', 'EpiB']
univar_str = bivar_str[univar_dim]

# explore data by plotting histograms
explore_data = True
num_bins = 10
colors = ['b', 'r']
alpha = 0.5

# ID of the measurement vector to use as an example
id_x_str = "MB83"

# options for outputting results
print_values = True  # printout example values given in paper
limX = [-7, 7]  # x-axis range for Tippett plots


## Load data

filename = "data/jfo14482-sup-0001-tables1.xlsx"
data_all = pd.read_excel(
    filename, engine='openpyxl', skiprows=1,
    usecols=[0, 1, 3, 4]
)
data_all["Individual"] = data_all["Individual"].map(lambda x:x.replace('*', ''))
datacols = data_all.columns[2:4]
data = data_all[datacols]

II_male = data_all[data_all.Sex == 'M'].index
II_female = data_all[data_all.Sex == 'F'].index

data_male = data.iloc[II_male]
data_female = data.iloc[II_female]

data_male_centred = data_male - data_male.mean()
data_female_centred = data_female - data_female.mean()

num_male = data_male.shape[0]
num_female = data_female.shape[0]
num_spk = data.shape[0]


# Explore data (histograms)
if explore_data:
    for ii in range(len(datacols)):
        fig, ax = plt.subplots(figsize=(8, 6))
        datacol = datacols[ii]
        _, _, _ = ax.hist(
            [data_male[datacol], data_female[datacol]], 
            bins=num_bins,
            histtype='stepfilled',
            color=colors,
            edgecolor='k',
            alpha=alpha,
            linewidth=1,
            density=True
        )

        # Fix to ensure the legend is correct (possible Matplotlib error)
        handles = [plt.Rectangle((0, 0), 1, 1, color=cc, ec="k",
                   alpha=alpha) for cc in colors]
        _ = ax.set_xlabel(bivar_str[ii] + " (mm)")
        _ = ax.legend(handles, ['M', 'F'])
        _ = ax.set_ylabel("probability density")


## Data with example pulled out

II_x = data_all[data_all.Individual == id_x_str].index
II_all_x = data_all.index.difference(II_x)
II_male_x = II_male.difference(II_x)
II_female_x = II_female.difference(II_x)

data_test_x = data.iloc[II_x].values[0]
data_male_x = data.iloc[II_male_x]
data_female_x = data.iloc[II_female_x]

data_male_x_centred = data_male_x - data_male_x.mean()
data_female_x_centred = data_female_x - data_female_x.mean()

num_male_x = data_male_x.shape[0]
num_female_x = data_female_x.shape[0]


## Linear discriminant analysis 1D

# Example
data_1d_test_x = float(data_test_x[univar_dim])

mu_1d_m_x = data_male_x[univar_str].mean()
mu_1d_f_x = data_female_x[univar_str].mean()

sigma_1d_pooled_x = np.std(
    np.concatenate(
        (data_male_x_centred[univar_str], data_female_x_centred[univar_str])
    ),
    ddof=1
)

L_1d_lda_m_x = norm(mu_1d_m_x, sigma_1d_pooled_x).pdf(data_1d_test_x)
L_1d_lda_f_x = norm(mu_1d_f_x, sigma_1d_pooled_x).pdf(data_1d_test_x)

LR_1d_lda_x = L_1d_lda_m_x / L_1d_lda_f_x


# Linear equation
a_lda_x = (mu_1d_f_x ** 2 - mu_1d_m_x ** 2) / (2 * sigma_1d_pooled_x ** 2)
b_lda_x = (mu_1d_m_x - mu_1d_f_x) / sigma_1d_pooled_x ** 2

logLR_1d_lda_lineq_x = a_lda_x + b_lda_x * data_1d_test_x
LR_1d_lda_lineq_x = np.exp(logLR_1d_lda_lineq_x)


# Leave-one-out cross-validation
mu_1d_m = data_male[univar_str].mean()
mu_1d_f = data_female[univar_str].mean()

LR_1d_lda_m = np.empty(len(II_male)) * np.nan
LR_1d_lda_f = np.empty(len(II_female)) * np.nan

for ii in range(len(II_male)):
    ii_idx = pd.Index([II_male[ii]])
    ii_train_idx = II_male.difference(ii_idx)
    
    data_test = data.loc[ii_idx][univar_str]
    data_train = data_male.loc[ii_train_idx][univar_str]
    data_train_centred = data_male_centred.loc[ii_train_idx][univar_str]

    mu_1d_m_temp = data_train.mean()
    sigma_1d_pooled_temp = np.std(
        np.concatenate(
            (data_train_centred, data_female_centred[univar_str])
        ),
        ddof=1
    )
    
    g_1d_m_lda = norm(mu_1d_m_temp, sigma_1d_pooled_temp)
    g_1d_f_lda = norm(mu_1d_f, sigma_1d_pooled_temp)

    LR_1d_lda_m[ii] = g_1d_m_lda.pdf(data_test) / g_1d_f_lda.pdf(data_test)

for ii in range(len(II_female)):
    ii_idx = pd.Index([II_female[ii]])
    ii_train_idx = II_female.difference(ii_idx)
    
    data_test = data.loc[ii_idx][univar_str]
    data_train = data_female.loc[ii_train_idx][univar_str]
    data_train_centred = data_female_centred.loc[ii_train_idx][univar_str]

    mu_1d_f_temp = data_train.mean()
    sigma_1d_pooled_temp = np.std(
        np.concatenate(
            (data_train_centred, data_male_centred[univar_str])
        ),
        ddof=1
    )
    
    g_1d_m_lda = norm(mu_1d_m, sigma_1d_pooled_temp)
    g_1d_f_lda = norm(mu_1d_f_temp, sigma_1d_pooled_temp)

    LR_1d_lda_f[ii] = g_1d_m_lda.pdf(data_test) / g_1d_f_lda.pdf(data_test)

Cllr_1d_lda = Cllr(LR_1d_lda_m, LR_1d_lda_f)


# Print example values given in paper + Tippett plot
if print_values:
    print(f"\n________________________________\nLDA 1D:\n")
    print(f"data_1d_test_x:        {data_1d_test_x:.5g}")
    print()
    print(f"mu_1d_m_x:             {mu_1d_m_x:.5g}")
    print(f"mu_1d_f_x:             {mu_1d_f_x:.5g}")
    print()
    print(f"sigma_1d_pooled_x:      {sigma_1d_pooled_x:.5g}")
    print()
    print(f"L_1d_lda_m_x:           {L_1d_lda_m_x:.5g}")
    print(f"L_1d_lda_f_x:           {L_1d_lda_f_x:.5g}")
    print()
    print(f"LR_1d_lda_x:            {LR_1d_lda_x:.5g}")
    print()
    print(f"a_lda_x:              {a_lda_x:.5g}")
    print(f"b_lda_x:                {b_lda_x:.5g}")
    print()
    print(f"logLR_1d_lda_lineq_x:   {logLR_1d_lda_lineq_x:.5g}")
    print(f"LR_1d_lda_lineq_x:      {LR_1d_lda_lineq_x:.5g}")
    print()
    print(f"Cllr_1d_lda:            {Cllr_1d_lda:.5g}")
    print("________________________________\n")

    plot_tippett(
        tar_llrs=LR_1d_lda_m,
        nontar_llrs=LR_1d_lda_f,
        xlim=limX,
        title="LDA 1D"
    )


## Linear discriminant analysis 2D

# Example
mu_2d_m_x = data_male_x.mean().values
mu_2d_f_x = data_female_x.mean().values

sigma_2d_pooled_x = np.cov(
    np.concatenate(
        (data_male_x_centred, data_female_x_centred)
    ),
    rowvar=False,
    ddof=1
)

L_2d_lda_m_x = multivariate_normal(
                   mean=mu_2d_m_x, cov=sigma_2d_pooled_x
               ).pdf(data_test_x)
L_2d_lda_f_x = multivariate_normal(
                   mean=mu_2d_f_x, cov=sigma_2d_pooled_x
               ).pdf(data_test_x)

LR_2d_lda_x = L_2d_lda_m_x / L_2d_lda_f_x


# Leave-one-out cross-validation
mu_2d_m = data_male.mean()
mu_2d_f = data_female.mean()

LR_2d_lda_m = np.empty(len(II_male)) * np.nan
LR_2d_lda_f = np.empty(len(II_female)) * np.nan

for ii in range(len(II_male)):
    ii_idx = pd.Index([II_male[ii]])
    ii_train_idx = II_male.difference(ii_idx)
    
    data_test = data.loc[ii_idx]
    data_train = data_male.loc[ii_train_idx]
    data_train_centred = data_male_centred.loc[ii_train_idx]

    mu_2d_m_temp = data_train.mean()
    sigma_2d_pooled_temp = np.cov(
        np.concatenate(
            (data_train_centred, data_female_centred)
        ),
        rowvar=False,
        ddof=1
    )
    
    g_2d_m_lda = multivariate_normal(
        mean=mu_2d_m_temp, cov=sigma_2d_pooled_temp
    )
    g_2d_f_lda = multivariate_normal(
        mean=mu_2d_f, cov=sigma_2d_pooled_temp
    )
    LR_2d_lda_m[ii] = g_2d_m_lda.pdf(data_test) / g_2d_f_lda.pdf(data_test)

for ii in range(len(II_female)):
    ii_idx = pd.Index([II_female[ii]])
    ii_train_idx = II_female.difference(ii_idx)
    
    data_test = data.loc[ii_idx]
    data_train = data_female.loc[ii_train_idx]
    data_train_centred = data_female_centred.loc[ii_train_idx]

    mu_2d_f_temp = data_train.mean()
    sigma_2d_pooled_temp = np.cov(
        np.concatenate(
            (data_train_centred, data_male_centred)
        ),
        rowvar=False,
        ddof=1
    )
    
    g_2d_m_lda = multivariate_normal(
        mean=mu_2d_m, cov=sigma_2d_pooled_temp
    )
    g_2d_f_lda = multivariate_normal(
        mean=mu_2d_f_temp, cov=sigma_2d_pooled_temp
    )
    LR_2d_lda_f[ii] = g_2d_m_lda.pdf(data_test) / g_2d_f_lda.pdf(data_test)

Cllr_2d_lda = Cllr(LR_2d_lda_m, LR_2d_lda_f)


# Print example values given in paper + Tippett plot
if print_values:
    print(f"\n________________________________\nLDA 2D:\n")
    print(f"data_test_x:           {data_test_x[0]:.5g}    {data_test_x[1]:.5g}")
    print()
    print(f"mu_2d_m_x:             {mu_2d_m_x[0]:.5g}   {mu_2d_m_x[1]:.5g}")
    print(f"mu_2d_f_x:             {mu_2d_f_x[0]:.5g}   {mu_2d_f_x[1]:.5g}")
    print()
    print(f"sigma_2d_pooled_x:      {sigma_2d_pooled_x[0][0]:.5g}   {sigma_2d_pooled_x[0][1]:.5g}")
    print(f"                        {sigma_2d_pooled_x[1][0]:.5g}  {sigma_2d_pooled_x[1][1]:.5g}")
    print()
    print(f"L_2d_lda_m_x:           {L_2d_lda_m_x:.5g}")
    print(f"L_2d_lda_f_x:           {L_2d_lda_f_x:.5g}")
    print()
    print(f"LR_2d_lda_x:            {LR_2d_lda_x:.5g}")
    print()
    print(f"Cllr_2d_lda:            {Cllr_2d_lda:.5g}")
    print("________________________________\n")

    plot_tippett(
        tar_llrs=LR_2d_lda_m,
        nontar_llrs=LR_2d_lda_f,
        xlim=limX,
        title="LDA 2D"
    )


## Quadratic discriminant analysis 1D

# Example
sigma_1d_m_x = data_male_x[univar_str].std(ddof=1)
sigma_1d_f_x = data_female_x[univar_str].std(ddof=1)

L_1d_qda_m_x = norm(mu_1d_m_x, sigma_1d_m_x).pdf(data_1d_test_x)
L_1d_qda_f_x = norm(mu_1d_f_x, sigma_1d_f_x).pdf(data_1d_test_x)

LR_1d_qda_x = L_1d_qda_m_x / L_1d_qda_f_x


# Leave-one-out cross-validation
sigma_1d_m = data_male[univar_str].std()
sigma_1d_f = data_female[univar_str].std()

LR_1d_qda_m = np.empty(len(II_male)) * np.nan
LR_1d_qda_f = np.empty(len(II_female)) * np.nan

for ii in range(len(II_male)):
    ii_idx = pd.Index([II_male[ii]])
    ii_train_idx = II_male.difference(ii_idx)
    
    data_test = data.loc[ii_idx][univar_str]
    data_train = data_male.loc[ii_train_idx][univar_str]
    data_train_centred = data_male_centred.loc[ii_train_idx][univar_str]

    mu_1d_m_temp = data_train.mean()
    sigma_1d_m_temp = data_train.std(ddof=1)
    
    g_1d_m_qda = norm(mu_1d_m_temp, sigma_1d_m_temp)
    g_1d_f_qda = norm(mu_1d_f, sigma_1d_f)

    LR_1d_qda_m[ii] = g_1d_m_qda.pdf(data_test) / g_1d_f_qda.pdf(data_test)

for ii in range(len(II_female)):
    ii_idx = pd.Index([II_female[ii]])
    ii_train_idx = II_female.difference(ii_idx)
    
    data_test = data.loc[ii_idx][univar_str]
    data_train = data_female.loc[ii_train_idx][univar_str]
    data_train_centred = data_female_centred.loc[ii_train_idx][univar_str]

    mu_1d_f_temp = data_train.mean()
    sigma_1d_f_temp = data_train.std(ddof=1)
    
    g_1d_m_qda = norm(mu_1d_m, sigma_1d_m)
    g_1d_f_qda = norm(mu_1d_f_temp, sigma_1d_f_temp)

    LR_1d_qda_f[ii] = g_1d_m_qda.pdf(data_test) / g_1d_f_qda.pdf(data_test)

Cllr_1d_qda = Cllr(LR_1d_qda_m, LR_1d_qda_f)


# Print example values given in paper + Tippett plot
if print_values:
    print(f"\n________________________________\nQDA 1D:\n")
    print(f"data_1d_test_x:        {data_1d_test_x:.5g}")
    print()
    print(f"mu_1d_m_x:             {mu_1d_m_x:.5g}")
    print(f"mu_1d_f_x:             {mu_1d_f_x:.5g}")
    print()
    print(f"sigma_1d_m_x:           {sigma_1d_m_x:.5g}")
    print(f"sigma_1d_f_x:           {sigma_1d_f_x:.5g}")
    print()
    print(f"L_1d_qda_m_x:           {L_1d_qda_m_x:.5g}")
    print(f"L_1d_qda_f_x:           {L_1d_qda_f_x:.5g}")
    print()
    print(f"LR_1d_qda_x:            {LR_1d_qda_x:.5g}")
    print()
    print(f"Cllr_1d_qda:            {Cllr_1d_qda:.5g}")
    print("________________________________\n")

    plot_tippett(
        tar_llrs=LR_1d_qda_m,
        nontar_llrs=LR_1d_qda_f,
        xlim=limX,
        title="QDA 1D"
    )


## Quadratic discriminant analysis 2D

# Example
mu_2d_m_x = data_male_x.mean().values
mu_2d_f_x = data_female_x.mean().values

sigma_2d_m_x = np.cov(data_male_x_centred, rowvar=False, ddof=1)
sigma_2d_f_x = np.cov(data_female_x_centred, rowvar=False, ddof=1)

L_2d_qda_m_x = multivariate_normal(
                   mean=mu_2d_m_x, cov=sigma_2d_m_x
               ).pdf(data_test_x)
L_2d_qda_f_x = multivariate_normal(
                   mean=mu_2d_f_x, cov=sigma_2d_f_x
               ).pdf(data_test_x)

LR_2d_qda_x = L_2d_qda_m_x / L_2d_qda_f_x


# Leave-one-out cross-validation
sigma_2d_m = np.cov(data_male, rowvar=False, ddof=1)
sigma_2d_f = np.cov(data_female, rowvar=False, ddof=1)

LR_2d_qda_m = np.empty(len(II_male)) * np.nan
LR_2d_qda_f = np.empty(len(II_female)) * np.nan

for ii in range(len(II_male)):
    ii_idx = pd.Index([II_male[ii]])
    ii_train_idx = II_male.difference(ii_idx)
    
    data_test = data.loc[ii_idx]
    data_train = data_male.loc[ii_train_idx]
    data_train_centred = data_male_centred.loc[ii_train_idx]

    mu_2d_m_temp = data_train.mean()
    sigma_2d_m_temp = np.cov(data_train, rowvar=False, ddof=1)
    
    g_2d_m_qda = multivariate_normal(mean=mu_2d_m_temp, cov=sigma_2d_m_temp)
    g_2d_f_qda = multivariate_normal(mean=mu_2d_f, cov=sigma_2d_f)

    LR_2d_qda_m[ii] = g_2d_m_qda.pdf(data_test) / g_2d_f_qda.pdf(data_test)

for ii in range(len(II_female)):
    ii_idx = pd.Index([II_female[ii]])
    ii_train_idx = II_female.difference(ii_idx)
    
    data_test = data.loc[ii_idx]
    data_train = data_female.loc[ii_train_idx]
    data_train_centred = data_female_centred.loc[ii_train_idx]

    mu_2d_f_temp = data_train.mean()
    sigma_2d_f_temp = np.cov(data_train, rowvar=False, ddof=1)
    
    g_2d_m_qda = multivariate_normal(mean=mu_2d_m, cov=sigma_2d_m)
    g_2d_f_qda = multivariate_normal(mean=mu_2d_f_temp, cov=sigma_2d_f_temp)

    LR_2d_qda_f[ii] = g_2d_m_qda.pdf(data_test) / g_2d_f_qda.pdf(data_test)

Cllr_2d_qda = Cllr(LR_2d_qda_m, LR_2d_qda_f)


# Print example values given in paper + Tippett plot
if print_values:
    print(f"\n________________________________\nQDA 2D:\n")
    print(f"data_test_x:           {data_test_x[0]:.5g}    {data_test_x[1]:.5g}")
    print()
    print(f"mu_2d_m_x:             {mu_2d_m_x[0]:.5g}   {mu_2d_m_x[1]:.5g}")
    print(f"mu_2d_f_x:             {mu_2d_f_x[0]:.5g}   {mu_2d_f_x[1]:.5g}")
    print()
    print(f"sigma_2d_m_x:           {sigma_2d_m_x[0][0]:.5g}   {sigma_2d_m_x[0][1]:.5g}")
    print(f"                        {sigma_2d_m_x[1][0]:.5g}  {sigma_2d_m_x[1][1]:.5g}")
    print(f"sigma_2d_f_x:           {sigma_2d_f_x[0][0]:.5g}   {sigma_2d_f_x[0][1]:.5g}")
    print(f"                        {sigma_2d_f_x[1][0]:.5g}    {sigma_2d_f_x[1][1]:.5g}")
    print()
    print(f"L_2d_qda_m_x:           {L_2d_qda_m_x:.5g}")
    print(f"L_2d_qda_f_x:           {L_2d_qda_f_x:.5g}")
    print()
    print(f"LR_2d_qda_x:            {LR_2d_qda_x:.5g}")
    print()
    print(f"Cllr_2d_qda:            {Cllr_2d_qda:.5g}")
    print("________________________________\n")

    plot_tippett(
        tar_llrs=LR_2d_qda_m,
        nontar_llrs=LR_2d_qda_f,
        xlim=limX,
        title="QDA 2D"
    )

## Logistic Regression

# Example (1D)
logreg_data_1d = data.loc[II_all_x][univar_str].values[:, np.newaxis]
logreg_target = np.array((data_all.loc[II_all_x].Sex == 'F').values, dtype=int)
logreg_model_1d = LogisticRegression(solver="newton-cg", penalty="none")
_ = logreg_model_1d.fit(X=logreg_data_1d, y=logreg_target)

a_LogReg_x = -logreg_model_1d.intercept_[0]
b_LogReg_x = -logreg_model_1d.coef_[0][0]

logLR_1d_LogReg_x = a_LogReg_x + b_LogReg_x * data_1d_test_x
LR_1d_LogReg_x = np.exp(logLR_1d_LogReg_x)


# Example (2D)
logreg_data_2d = data.loc[II_all_x].values
logreg_model_2d = LogisticRegression(solver="newton-cg", penalty="none")
_ = logreg_model_2d.fit(X=logreg_data_2d, y=logreg_target)

coefs_2d_LogReg_x = np.concatenate(
    (-logreg_model_2d.intercept_, -logreg_model_2d.coef_[0])
)
logLR_2d_LogReg_x = coefs_2d_LogReg_x @ \
                    np.concatenate((np.ones(1), data_test_x))
LR_2d_LogReg_x = np.exp(logLR_2d_LogReg_x)


# Leave-one-out cross-validation (1D)
LR_1d_LogReg = np.zeros(len(data_all.index))
for ii in range(len(data_all.index)):
    ii_idx = pd.Index([ii])
    data_test = data.loc[ii_idx][univar_str].values[0]
    
    II_data_train = data.index.difference(ii_idx)
    data_train = data.loc[II_data_train][univar_str].values[:, np.newaxis]
    
    logreg_target = np.array(
        (data_all.Sex == 'F').loc[II_data_train].values, dtype=int
    )

    logreg_1d = LogisticRegression(solver="newton-cg", penalty="none")
    logreg_1d.fit(X=data_train, y=logreg_target)
    
    p_m, p_f = logreg_1d.predict_log_proba([[data_test]])[0]
    LR_1d_LogReg[ii] = np.exp(p_m - p_f)

data_all["LR_1d_LogReg"] = LR_1d_LogReg

LR_1d_LogReg_m = data_all.LR_1d_LogReg[data_all.Sex == 'M']
LR_1d_LogReg_f = data_all.LR_1d_LogReg[data_all.Sex == 'F']
Cllr_1d_LogReg = Cllr(LR_1d_LogReg_m, LR_1d_LogReg_f)


# Leave-one-out cross-validation (2D)
LR_2d_LogReg = np.zeros(len(data_all.index))
for ii in range(len(data_all.index)):
    ii_idx = pd.Index([ii])
    data_test = data.loc[ii_idx].values

    II_data_train = data.index.difference(ii_idx)
    data_train = data.loc[II_data_train].values
    
    logreg_target = np.array(
        (data_all.Sex == 'F').loc[II_data_train].values, dtype=int
    )

    logreg_2d = LogisticRegression(solver="newton-cg", penalty="none")
    logreg_2d.fit(X=data_train, y=logreg_target)
    
    p_m, p_f = logreg_2d.predict_log_proba(data_test)[0]
    LR_2d_LogReg[ii] = np.exp(p_m - p_f)

data_all["LR_2d_LogReg"] = LR_2d_LogReg

LR_2d_LogReg_m = data_all.LR_2d_LogReg[data_all.Sex == 'M']
LR_2d_LogReg_f = data_all.LR_2d_LogReg[data_all.Sex == 'F']
Cllr_2d_LogReg = Cllr(LR_2d_LogReg_m, LR_2d_LogReg_f)


# Print example values given in paper + Tippett plot
if print_values:
    print(f"\n________________________________\nLogReg 1D:\n")
    print(f"data_1d_test_x:        {data_1d_test_x:.5g}")
    print()
    print(f"a_LogReg_x:           {a_LogReg_x:.5g}")
    print(f"b_LogReg_x:             {b_LogReg_x:.5g}")
    print()
    print(f"logLR_1d_LogReg_x:      {logLR_1d_LogReg_x:.5g}")
    print(f"LR_1d_LogReg_x:         {LR_1d_LogReg_x:.5g}")
    print()
    print(f"Cllr_1d_LogReg:         {Cllr_1d_LogReg:.5g}")
    print("________________________________\n")

    plot_tippett(
        tar_llrs=LR_1d_LogReg_m,
        nontar_llrs=LR_1d_LogReg_f,
        xlim=limX,
        title="LogReg 1D"
    )

if print_values:
    print(f"\n________________________________\nLogReg 2D:\n")
    print(f"data_test_x:           {data_test_x[0]:.5g}    {data_test_x[1]:.5g}")
    print()
    print(f"coefs_2d_LogReg_x:    {coefs_2d_LogReg_x[0]:.5g}")
    print(f"                        {coefs_2d_LogReg_x[1]:.5g}")
    print(f"                        {coefs_2d_LogReg_x[2]:.5g}")
    print(f"logLR_2d_LogReg_x:      {logLR_2d_LogReg_x:.5g}")
    print(f"LR_2d_LogReg_x:         {LR_2d_LogReg_x:.5g}")
    print()
    print(f"Cllr_2d_LogReg:         {Cllr_2d_LogReg:.5g}")
    print("________________________________\n")

    plot_tippett(
        tar_llrs=LR_2d_LogReg_m,
        nontar_llrs=LR_2d_LogReg_f,
        xlim=limX,
        title="LogReg 2D"
    )

    plt.show()
